/*
 * Copyright 2015 Andrew T. Flowers, K0SM
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package dopplerpsk.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * SQF parser parses the doppler.sqf file to find uplink/downlinnk frequencies.
 * The format is identical to that used by SatPC32:
 * 
 * name,downlinkKHz,uplinkKHz,downlinkMode,uplinkMode,NOR/REV,[comment]
 * 
 * e.g.,
 * 
 * 
 * NO-44,145825,145825,PKT,PKT,NOR,0,0,1k2 AFSK
 * HO-68,435740,145950.8,CW,CW,REV,0,0
 * ;This is a comment and will be ignored
 *
 * @author Andrew T. Flowers
 */
public class SqfParser {

    private final Map<String, Sqf> sqfMap;

    private final File file;

    /**
     * Creats a new SQF file parser.
     * 
     * @param f the file to be parsed (typically doppler.sqf)
     */
    public SqfParser(File f) {
        file = f;
        sqfMap = new TreeMap();
    }

    /**
     * Loads the Doppler information into memory.
     * 
     * @throws FileNotFoundException 
     */
    public void parse() throws FileNotFoundException {
        BufferedReader br = null;
        try {
            // read in file line by line
            // ignore line started with semicolon
            // validate 8-9 comma-delimited fields:
            //name,downlinkKHz,uplinkKHz,downlinkMode,uplinkMode,NOR/REV,[comment]

            br = new BufferedReader(new FileReader(file));
            String line = br.readLine();
            while (line != null) {
                line = line.trim();

                
                try {
                    if (!line.startsWith(";") && line.length() > 0) {
                        String[] strs = line.split(",");
                        String name = strs[0];
                        double downlinkHz = Double.parseDouble(strs[1]) * 1000.0;
                        double uplinkHz = Double.parseDouble(strs[2]) * 1000.0;
                        boolean isRev = "REV".equalsIgnoreCase(strs[5]);

                        Sqf sqf = new Sqf(name, downlinkHz, uplinkHz, isRev);
                        System.out.println(sqf.getName());
                        sqfMap.put(name, sqf);
                    }
                } catch (Exception e) {
                    System.out.println("Error reading sqf line: " + line);
                    e.printStackTrace();
                }
                line = br.readLine();
            }
        } catch (IOException ex) {
            Logger.getLogger(SqfParser.class.getName()).log(Level.SEVERE, null, ex);
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException ex) {
                    Logger.getLogger(SqfParser.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
    }

    /**
     * Gets the frequency information for a particular satellite, or null if
     * the referenced satellite does not exist.
     * 
     * @param name the name of the satellite
     * @return frequency information for the satellite or null if satellite does
     * not exist.
     */
    public Sqf getSqf(String name) {
        return sqfMap.get(name);
    }

    
}
