/*
 * Copyright 2015 Andrew T. Flowers, K0SM
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package dopplerpsk.ui;

import java.text.DecimalFormat;
import javax.swing.JOptionPane;

/**
 * UI for editing station information.
 * 
 * @author Andrew T. Flowers
 */
public class StationInfoDialog extends javax.swing.JDialog {

   /**
     * Indicates the user canceled the satellite selection.
     */
    public static final int CANCEL_ACTION = 0;
    
    /**
     * Indicates the user pressed "OK"--the satellite name should
     * be retrieved with getSelectedSatelliteName().
     */
    public static final int OK_ACTION = 1;
    
    private static final DecimalFormat degFormat = new DecimalFormat("0.000###");
    private int userAction = 0;
    private double lat;
    private double lon;
    private double alt;

    /**
     * Creates new form SatelliteChooserDialog
     */
    public StationInfoDialog(java.awt.Frame parent, boolean modal) {
        super(parent, modal);
        initComponents();
    }

    /**
     * Creates a new StationInfoDialog
     * @param parent parent window
     * @param latitude current station latitude in degrees
     * @param longitude current station longitude in degrees
     * @param altitude current station altitude in meters ASL
     * @param modal whether or not this is a modal dialog.
     */
    public StationInfoDialog(java.awt.Frame parent, double latitude, double longitude, double altitude, boolean modal) {
        super(parent, modal);
        initComponents();
        
        lat = latitude;
        lon = longitude;
        alt = altitude;
        
        if (lat < 0) {
            latTextField.setText(degFormat.format(-lat));
            southRadioButton.setSelected(true);
        } else {
           latTextField.setText(degFormat.format(lat)); 
           northRadioButton.setSelected(true);
        }
        
        if (longitude < 0) {
            lonTextField.setText(degFormat.format(-lon));
            westRadioButton.setSelected(true);
        } else {
           lonTextField.setText(degFormat.format(lon)); 
           eastRadioButton.setSelected(true);
        }
        altTextField.setText("" + (int) Math.round(alt));
        

    }

    
    /**
     * Get the user action (OK_ACTION or CANCEL_ACTION)
     * @return OK_ACTION or CANCEL_ACTION
     */
    public int getUserAction() {
        return userAction;
    }

    /**
     * @return the latitude in degrees
     */
    public double getLat() {
        return lat;
    }

    /**
     * @return the longitude in degrees
     */
    public double getLon() {
        return lon;
    }

    /**
     * @return the altitude in meters ASL
     */
    public double getAlt() {
        return alt;
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        latButtonGroup = new javax.swing.ButtonGroup();
        lonButtonGroup = new javax.swing.ButtonGroup();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();
        jPanel1 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        jLabel3 = new javax.swing.JLabel();
        latTextField = new javax.swing.JTextField();
        lonTextField = new javax.swing.JTextField();
        altTextField = new javax.swing.JTextField();
        northRadioButton = new javax.swing.JRadioButton();
        southRadioButton = new javax.swing.JRadioButton();
        eastRadioButton = new javax.swing.JRadioButton();
        westRadioButton = new javax.swing.JRadioButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Set Station Location");

        okButton.setText("OK");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        jLabel1.setText("Latitude:");

        jLabel2.setText("Longitude:");

        jLabel3.setText("Altitude (meters)");

        latTextField.setHorizontalAlignment(javax.swing.JTextField.TRAILING);
        latTextField.setText("0.0");

        lonTextField.setHorizontalAlignment(javax.swing.JTextField.TRAILING);
        lonTextField.setText("0.0");

        altTextField.setHorizontalAlignment(javax.swing.JTextField.TRAILING);
        altTextField.setText("0.0");

        latButtonGroup.add(northRadioButton);
        northRadioButton.setText("N");

        latButtonGroup.add(southRadioButton);
        southRadioButton.setText("S");

        lonButtonGroup.add(eastRadioButton);
        eastRadioButton.setText("E");

        lonButtonGroup.add(westRadioButton);
        westRadioButton.setText("W");

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jLabel2)
                    .addComponent(jLabel3)
                    .addComponent(jLabel1))
                .addGap(18, 18, 18)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(latTextField)
                    .addComponent(altTextField, javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, jPanel1Layout.createSequentialGroup()
                        .addComponent(lonTextField, javax.swing.GroupLayout.PREFERRED_SIZE, 102, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(0, 0, Short.MAX_VALUE)))
                .addGap(18, 18, 18)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addComponent(northRadioButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(southRadioButton))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addComponent(eastRadioButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(westRadioButton)))
                .addGap(31, 31, 31))
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(latTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(northRadioButton)
                    .addComponent(southRadioButton))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel2)
                    .addComponent(lonTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(eastRadioButton)
                    .addComponent(westRadioButton))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel3)
                    .addComponent(altTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(cancelButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(okButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(okButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(cancelButton)
                        .addGap(0, 44, Short.MAX_VALUE)))
                .addContainerGap())
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed

        double latitude;
        double longitude;
        double altitude;
        try {
            latitude = Double.parseDouble(latTextField.getText());
            if (latitude < 0.0) {
                JOptionPane.showMessageDialog(rootPane, "Latitude cannot be negative.");
                return;
            }
            if (southRadioButton.isSelected()) {
                latitude = -latitude;
            }
        } catch (NumberFormatException nfe) {
            nfe.printStackTrace();
            JOptionPane.showMessageDialog(rootPane, "Latitude is not a valid number.");
            return;
        }

        try {
            longitude = Double.parseDouble(lonTextField.getText());
            if (longitude < 0.0) {
                JOptionPane.showMessageDialog(rootPane, "Longitude cannot be negative.");
                return;
            }
            if (westRadioButton.isSelected()) {
                longitude = -longitude;
            }

        } catch (NumberFormatException nfe) {
            nfe.printStackTrace();
            JOptionPane.showMessageDialog(rootPane, "Longitude is not a valid number.");
            return;
        }

        try {
            altitude = Double.parseDouble(altTextField.getText());
        } catch (NumberFormatException nfe) {
            nfe.printStackTrace();
            JOptionPane.showMessageDialog(rootPane, "Altitude is not a valid number.");
            return;
        }

        // if we get here the input is valid
        userAction = OK_ACTION;
        lat = latitude;
        lon = longitude;
        alt = altitude;

        this.dispose();
    }//GEN-LAST:event_okButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed
        userAction = CANCEL_ACTION;
        this.dispose();
    }//GEN-LAST:event_cancelButtonActionPerformed

    
    

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTextField altTextField;
    private javax.swing.JButton cancelButton;
    private javax.swing.JRadioButton eastRadioButton;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JPanel jPanel1;
    private javax.swing.ButtonGroup latButtonGroup;
    private javax.swing.JTextField latTextField;
    private javax.swing.ButtonGroup lonButtonGroup;
    private javax.swing.JTextField lonTextField;
    private javax.swing.JRadioButton northRadioButton;
    private javax.swing.JButton okButton;
    private javax.swing.JRadioButton southRadioButton;
    private javax.swing.JRadioButton westRadioButton;
    // End of variables declaration//GEN-END:variables

}
