/*
 * Copyright 2015 Andrew T. Flowers, K0SM
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package dopplerpsk.event;

/**
 * Events containing new satellite position and Doppler information.
 * 
 * @author Andrew T. Flowers
 */
public class SatelliteUpdateEvent {
    private final double az;
    private final double el;
    private final double df;
    private final double ddf;
    private final String name;
    
    /**
     * Creates a new SatelliteUpdateEvent.
     * 
     * @param name the name of the satellite
     * @param az azimuth in degrees
     * @param el elevation in degrees
     * @param df Doppler shift in Hz
     * @param ddf Doppler shift *rate* in Hz/s
     */
    public SatelliteUpdateEvent(String name, double az, double el, double df, double ddf) {
        this.name = name;
        this.az = az;
        this.el = el;
        this.df = df;
        this.ddf = ddf;
    }

    /**
     * 
     * @return name of the satellite
     */
    public String getName() {
        return name;
    }
    
    /**
     * @return the az Azimuth in degrees.
     */
    public double getAz() {
        return az;
    }

    /**
     * @return the el Elevation in degrees.
     */
    public double getEl() {
        return el;
    }

    /**
     * Gets the absolute Doppler shift in Hz.
     * @return Doppler shift in Hz 
     * 
     */
    public double getDopplerShift() {
        return df;
    }
    
    /**
     * Gets the Doppler rate in Hz/s.  This is how fast the signal is "drifting"
     * and in what direction.
     * 
     * @return Doppler shift rate in Hz/s
     */
    public double getDopplerRate() {
        return ddf;
    }
    
    @Override
    public String toString() {
        return ("name=" + name + " az=" + this.az + " el=" + this.el + " df =" + this.df + " ddf=" + this.ddf);
    }
}
