/*
 * Copyright 2015 Andrew T. Flowers, K0SM
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package dopplerpsk;

/**
 * Reprents a satellite, including two-line orbital elements, uplink/downlink 
 * frequencies, and transponder polarity.  This information is needed by
 * the ISatelliteEventSource implementations to calculate Doppler shift.
 * 
 * @author Andrew T. Flowers
 */
public class Satellite {
    
    private final String name;
    private final String tleLine1;
    private final String tleLine2;
    private final double upLinkHz;
    private final double downlinkHz;
    private final boolean inverting;
    
    /**
     * Create a new Satellite.
     * 
     * @param name name of the satellite
     * @param tleLine1 first data line of the TLEs
     * @param tleLine2 second data line of the TLEs
     * @param uplinkFreq uplink frequency in Hz
     * @param downlinkFreq downlink frequency in Hz
     * @param inverting  true if transponder is inverting
     */
    public Satellite(String name, String tleLine1, String tleLine2, double uplinkFreq, double downlinkFreq, boolean inverting) {
        this.name = name;
        this.tleLine1 = tleLine1;
        this.tleLine2 = tleLine2;
        this.upLinkHz = uplinkFreq;
        this.downlinkHz = downlinkFreq;
        this.inverting = inverting;
    }

    /**
     * The name is a the "key" by which a satellite is known.  It should be the
     * same in the input files (TLEs and Doppler.sqf).
     * 
     * @return the name of this satellite
     */
    public String getName() {
        return name;
    }

    /**
     * @return the first line of the TLEs for this satellite
     */
    public String getTleLine1() {
        return tleLine1;
    }

    /**
     * @return the second line of the TLEs for this satellite
     */
    public String getTleLine2() {
        return tleLine2;
    }

    /**
     * @return the uplink frequency in Hz
     */
    public double getUpLinkHz() {
        return upLinkHz;
    }

    /**
     * @return the downlink frequency in Hz
     */
    public double getDownlinkHz() {
        return downlinkHz;
    }

    /**
     * @return true if transponder is inverting
     */
    public boolean isInverting() {
        return inverting;
    }
    
}
