/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import java.util.Collection;
import org.apache.commons.math3.analysis.interpolation.HermiteInterpolator;
import org.apache.commons.math3.geometry.euclidean.threed.Vector3D;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.MathUtils;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class CircularOrbit
extends Orbit {
    private static final long serialVersionUID = 20141228L;
    private final double a;
    private final double ex;
    private final double ey;
    private final double i;
    private final double raan;
    private final double alphaV;
    private final boolean serializePV;

    public CircularOrbit(double a, double ex, double ey, double i, double raan, double alpha, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        if (ex * ex + ey * ey >= 1.0) {
            throw OrekitException.createIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = a;
        this.ex = ex;
        this.ey = ey;
        this.i = i;
        this.raan = raan;
        switch (type) {
            case MEAN: {
                this.alphaV = this.eccentricToTrue(this.meanToEccentric(alpha));
                break;
            }
            case ECCENTRIC: {
                this.alphaV = this.eccentricToTrue(alpha);
                break;
            }
            case TRUE: {
                this.alphaV = alpha;
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
        this.serializePV = false;
    }

    public CircularOrbit(double a, double ex, double ey, double i, double raan, double alpha, PositionAngle type, TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        if (ex * ex + ey * ey >= 1.0) {
            throw OrekitException.createIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = a;
        this.ex = ex;
        this.ey = ey;
        this.i = i;
        this.raan = raan;
        switch (type) {
            case MEAN: {
                this.alphaV = this.eccentricToTrue(this.meanToEccentric(alpha));
                break;
            }
            case ECCENTRIC: {
                this.alphaV = this.eccentricToTrue(alpha);
                break;
            }
            case TRUE: {
                this.alphaV = alpha;
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
        this.serializePV = true;
    }

    public CircularOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        double r = pvP.getNorm();
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        if (rV2OnMu > 2.0) {
            throw OrekitException.createIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = r / (2.0 - rV2OnMu);
        Vector3D momentum = pvCoordinates.getMomentum();
        this.i = Vector3D.angle((Vector3D)momentum, (Vector3D)Vector3D.PLUS_K);
        Vector3D node = Vector3D.crossProduct((Vector3D)Vector3D.PLUS_K, (Vector3D)momentum);
        this.raan = FastMath.atan2((double)node.getY(), (double)node.getX());
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double xP = pvP.getX();
        double yP = pvP.getY();
        double zP = pvP.getZ();
        double x2 = (xP * cosRaan + yP * sinRaan) / this.a;
        double y2 = ((yP * cosRaan - xP * sinRaan) * cosI + zP * sinI) / this.a;
        double eSE = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)(mu * this.a));
        double eCE = rV2OnMu - 1.0;
        double e2 = eCE * eCE + eSE * eSE;
        double f = eCE - e2;
        double g = FastMath.sqrt((double)(1.0 - e2)) * eSE;
        double aOnR = this.a / r;
        double a2OnR2 = aOnR * aOnR;
        this.ex = a2OnR2 * (f * x2 + g * y2);
        this.ey = a2OnR2 * (f * y2 - g * x2);
        double beta = 1.0 / (1.0 + FastMath.sqrt((double)(1.0 - this.ex * this.ex - this.ey * this.ey)));
        this.alphaV = this.eccentricToTrue(FastMath.atan2((double)(y2 + this.ey + eSE * beta * this.ex), (double)(x2 + this.ex - eSE * beta * this.ey)));
        this.serializePV = true;
    }

    public CircularOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public CircularOrbit(Orbit op) {
        super(op.getFrame(), op.getDate(), op.getMu());
        this.a = op.getA();
        this.i = op.getI();
        this.raan = FastMath.atan2((double)op.getHy(), (double)op.getHx());
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        double equiEx = op.getEquinoctialEx();
        double equiEy = op.getEquinoctialEy();
        this.ex = equiEx * cosRaan + equiEy * sinRaan;
        this.ey = equiEy * cosRaan - equiEx * sinRaan;
        this.alphaV = op.getLv() - this.raan;
        this.serializePV = false;
    }

    @Override
    public OrbitType getType() {
        return OrbitType.CIRCULAR;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getEquinoctialEx() {
        return this.ex * FastMath.cos((double)this.raan) - this.ey * FastMath.sin((double)this.raan);
    }

    @Override
    public double getEquinoctialEy() {
        return this.ey * FastMath.cos((double)this.raan) + this.ex * FastMath.sin((double)this.raan);
    }

    public double getCircularEx() {
        return this.ex;
    }

    public double getCircularEy() {
        return this.ey;
    }

    @Override
    public double getHx() {
        return FastMath.cos((double)this.raan) * FastMath.tan((double)(this.i / 2.0));
    }

    @Override
    public double getHy() {
        return FastMath.sin((double)this.raan) * FastMath.tan((double)(this.i / 2.0));
    }

    public double getAlphaV() {
        return this.alphaV;
    }

    public double getAlpha(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getAlphaM() : (type == PositionAngle.ECCENTRIC ? this.getAlphaE() : this.getAlphaV());
    }

    public double getAlphaE() {
        double epsilon = FastMath.sqrt((double)(1.0 - this.ex * this.ex - this.ey * this.ey));
        double cosAlphaV = FastMath.cos((double)this.alphaV);
        double sinAlphaV = FastMath.sin((double)this.alphaV);
        return this.alphaV + 2.0 * FastMath.atan((double)((this.ey * cosAlphaV - this.ex * sinAlphaV) / (epsilon + 1.0 + this.ex * cosAlphaV + this.ey * sinAlphaV)));
    }

    private double eccentricToTrue(double alphaE) {
        double epsilon = FastMath.sqrt((double)(1.0 - this.ex * this.ex - this.ey * this.ey));
        double cosAlphaE = FastMath.cos((double)alphaE);
        double sinAlphaE = FastMath.sin((double)alphaE);
        return alphaE + 2.0 * FastMath.atan((double)((this.ex * sinAlphaE - this.ey * cosAlphaE) / (epsilon + 1.0 - this.ex * cosAlphaE - this.ey * sinAlphaE)));
    }

    public double getAlphaM() {
        double alphaE = this.getAlphaE();
        return alphaE - this.ex * FastMath.sin((double)alphaE) + this.ey * FastMath.cos((double)alphaE);
    }

    private double meanToEccentric(double alphaM) {
        double alphaE = alphaM;
        double shift = 0.0;
        double alphaEMalphaM = 0.0;
        double cosAlphaE = FastMath.cos((double)alphaE);
        double sinAlphaE = FastMath.sin((double)alphaE);
        int iter = 0;
        do {
            double f2 = this.ex * sinAlphaE - this.ey * cosAlphaE;
            double f1 = 1.0 - this.ex * cosAlphaE - this.ey * sinAlphaE;
            double f0 = alphaEMalphaM - f2;
            double f12 = 2.0 * f1;
            shift = f0 * f12 / (f1 * f12 - f0 * f2);
            alphaE = alphaM + (alphaEMalphaM -= shift);
            cosAlphaE = FastMath.cos((double)alphaE);
            sinAlphaE = FastMath.sin((double)alphaE);
        } while (++iter < 50 && FastMath.abs((double)shift) > 1.0E-12);
        return alphaE;
    }

    @Override
    public double getE() {
        return FastMath.sqrt((double)(this.ex * this.ex + this.ey * this.ey));
    }

    @Override
    public double getI() {
        return this.i;
    }

    public double getRightAscensionOfAscendingNode() {
        return this.raan;
    }

    @Override
    public double getLv() {
        return this.alphaV + this.raan;
    }

    @Override
    public double getLE() {
        return this.getAlphaE() + this.raan;
    }

    @Override
    public double getLM() {
        return this.getAlphaM() + this.raan;
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        double equEx = this.getEquinoctialEx();
        double equEy = this.getEquinoctialEy();
        double hx = this.getHx();
        double hy = this.getHy();
        double lE = this.getLE();
        double hx2 = hx * hx;
        double hy2 = hy * hy;
        double factH = 1.0 / (1.0 + hx2 + hy2);
        double ux = (1.0 + hx2 - hy2) * factH;
        double uy = 2.0 * hx * hy * factH;
        double uz = -2.0 * hy * factH;
        double vx = uy;
        double vy = (1.0 - hx2 + hy2) * factH;
        double vz = 2.0 * hx * factH;
        double exey = equEx * equEy;
        double ex2 = equEx * equEx;
        double ey2 = equEy * equEy;
        double e2 = ex2 + ey2;
        double eta = 1.0 + FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / eta;
        double cLe = FastMath.cos((double)lE);
        double sLe = FastMath.sin((double)lE);
        double exCeyS = equEx * cLe + equEy * sLe;
        double x = this.a * ((1.0 - beta * ey2) * cLe + beta * exey * sLe - equEx);
        double y = this.a * ((1.0 - beta * ex2) * sLe + beta * exey * cLe - equEy);
        double factor = FastMath.sqrt((double)(this.getMu() / this.a)) / (1.0 - exCeyS);
        double xdot = factor * (-sLe + beta * equEy * exCeyS);
        double ydot = factor * (cLe - beta * equEx * exCeyS);
        Vector3D position = new Vector3D(x * ux + y * vx, x * uy + y * vy, x * uz + y * vz);
        double r2 = position.getNormSq();
        Vector3D velocity = new Vector3D(xdot * ux + ydot * vx, xdot * uy + ydot * vy, xdot * uz + ydot * vz);
        Vector3D acceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt((double)r2)), position);
        return new TimeStampedPVCoordinates(this.getDate(), position, velocity, acceleration);
    }

    @Override
    public CircularOrbit shiftedBy(double dt) {
        return new CircularOrbit(this.a, this.ex, this.ey, this.i, this.raan, this.getAlphaM() + this.getKeplerianMeanMotion() * dt, PositionAngle.MEAN, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
    }

    @Override
    public CircularOrbit interpolate(AbsoluteDate date, Collection<Orbit> sample) {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousRAAN = Double.NaN;
        double previousAlphaM = Double.NaN;
        for (Orbit orbit : sample) {
            double continuousAlphaM;
            double continuousRAAN;
            CircularOrbit circ = (CircularOrbit)OrbitType.CIRCULAR.convertType(orbit);
            if (previousDate == null) {
                continuousRAAN = circ.getRightAscensionOfAscendingNode();
                continuousAlphaM = circ.getAlphaM();
            } else {
                double dt = circ.getDate().durationFrom(previousDate);
                double keplerAM = previousAlphaM + circ.getKeplerianMeanMotion() * dt;
                continuousRAAN = MathUtils.normalizeAngle((double)circ.getRightAscensionOfAscendingNode(), (double)previousRAAN);
                continuousAlphaM = MathUtils.normalizeAngle((double)circ.getAlphaM(), (double)keplerAM);
            }
            previousDate = circ.getDate();
            previousRAAN = continuousRAAN;
            previousAlphaM = continuousAlphaM;
            interpolator.addSamplePoint(circ.getDate().durationFrom(date), (double[][])new double[][]{{circ.getA(), circ.getCircularEx(), circ.getCircularEy(), circ.getI(), continuousRAAN, continuousAlphaM}});
        }
        double[] interpolated = interpolator.value(0.0);
        return new CircularOrbit(interpolated[0], interpolated[1], interpolated[2], interpolated[3], interpolated[4], interpolated[5], PositionAngle.MEAN, this.getFrame(), date, this.getMu());
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        double[][] jacobian = new double[6][6];
        TimeStampedPVCoordinates pvc = this.getPVCoordinates();
        Vector3D position = pvc.getPosition();
        Vector3D velocity = pvc.getVelocity();
        double x = position.getX();
        double y = position.getY();
        double z = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double pv = Vector3D.dotProduct((Vector3D)position, (Vector3D)velocity);
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double v2 = velocity.getNormSq();
        double mu = this.getMu();
        double oOsqrtMuA = 1.0 / FastMath.sqrt((double)(mu * this.a));
        double rOa = r / this.a;
        double aOr = this.a / r;
        double aOr2 = this.a / r2;
        double a2 = this.a * this.a;
        double ex2 = this.ex * this.ex;
        double ey2 = this.ey * this.ey;
        double e2 = ex2 + ey2;
        double epsilon = FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / (1.0 + epsilon);
        double eCosE = 1.0 - rOa;
        double eSinE = pv * oOsqrtMuA;
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        CircularOrbit.fillHalfRow(2.0 * aOr * aOr2, position, jacobian[0], 0);
        CircularOrbit.fillHalfRow(2.0 * a2 / mu, velocity, jacobian[0], 3);
        Vector3D danP = new Vector3D(v2, position, -pv, velocity);
        Vector3D danV = new Vector3D(r2, velocity, -pv, position);
        double recip = 1.0 / pvc.getMomentum().getNorm();
        double recip2 = recip * recip;
        Vector3D dwXP = new Vector3D(recip, new Vector3D(0.0, vz, -vy), -recip2 * sinRaan * sinI, danP);
        Vector3D dwYP = new Vector3D(recip, new Vector3D(-vz, 0.0, vx), recip2 * cosRaan * sinI, danP);
        Vector3D dwZP = new Vector3D(recip, new Vector3D(vy, -vx, 0.0), -recip2 * cosI, danP);
        Vector3D dwXV = new Vector3D(recip, new Vector3D(0.0, -z, y), -recip2 * sinRaan * sinI, danV);
        Vector3D dwYV = new Vector3D(recip, new Vector3D(z, 0.0, -x), recip2 * cosRaan * sinI, danV);
        Vector3D dwZV = new Vector3D(recip, new Vector3D(-y, x, 0.0), -recip2 * cosI, danV);
        CircularOrbit.fillHalfRow(sinRaan * cosI, dwXP, -cosRaan * cosI, dwYP, -sinI, dwZP, jacobian[3], 0);
        CircularOrbit.fillHalfRow(sinRaan * cosI, dwXV, -cosRaan * cosI, dwYV, -sinI, dwZV, jacobian[3], 3);
        CircularOrbit.fillHalfRow(sinRaan / sinI, dwYP, cosRaan / sinI, dwXP, jacobian[4], 0);
        CircularOrbit.fillHalfRow(sinRaan / sinI, dwYV, cosRaan / sinI, dwXV, jacobian[4], 3);
        double u = x * cosRaan + y * sinRaan;
        double cv = -x * sinRaan + y * cosRaan;
        double v = cv * cosI + z * sinI;
        Vector3D duP = new Vector3D(cv * cosRaan / sinI, dwXP, cv * sinRaan / sinI, dwYP, 1.0, new Vector3D(cosRaan, sinRaan, 0.0));
        Vector3D duV = new Vector3D(cv * cosRaan / sinI, dwXV, cv * sinRaan / sinI, dwYV);
        Vector3D dvP = new Vector3D(-u * cosRaan * cosI / sinI + sinRaan * z, dwXP, -u * sinRaan * cosI / sinI - cosRaan * z, dwYP, cv, dwZP, 1.0, new Vector3D(-sinRaan * cosI, cosRaan * cosI, sinI));
        Vector3D dvV = new Vector3D(-u * cosRaan * cosI / sinI + sinRaan * z, dwXV, -u * sinRaan * cosI / sinI - cosRaan * z, dwYV, cv, dwZV);
        Vector3D dc1P = new Vector3D(aOr2 * (2.0 * eSinE * eSinE + 1.0 - eCosE) / r2, position, -2.0 * aOr2 * eSinE * oOsqrtMuA, velocity);
        Vector3D dc1V = new Vector3D(-2.0 * aOr2 * eSinE * oOsqrtMuA, position, 2.0 / mu, velocity);
        Vector3D dc2P = new Vector3D(aOr2 * eSinE * (eSinE * eSinE - (1.0 - e2)) / (r2 * epsilon), position, aOr2 * (1.0 - e2 - eSinE * eSinE) * oOsqrtMuA / epsilon, velocity);
        Vector3D dc2V = new Vector3D(aOr2 * (1.0 - e2 - eSinE * eSinE) * oOsqrtMuA / epsilon, position, eSinE / (mu * epsilon), velocity);
        double cof1 = aOr2 * (eCosE - e2);
        double cof2 = aOr2 * epsilon * eSinE;
        Vector3D dexP = new Vector3D(u, dc1P, v, dc2P, cof1, duP, cof2, dvP);
        Vector3D dexV = new Vector3D(u, dc1V, v, dc2V, cof1, duV, cof2, dvV);
        Vector3D deyP = new Vector3D(v, dc1P, -u, dc2P, cof1, dvP, -cof2, duP);
        Vector3D deyV = new Vector3D(v, dc1V, -u, dc2V, cof1, dvV, -cof2, duV);
        CircularOrbit.fillHalfRow(1.0, dexP, jacobian[1], 0);
        CircularOrbit.fillHalfRow(1.0, dexV, jacobian[1], 3);
        CircularOrbit.fillHalfRow(1.0, deyP, jacobian[2], 0);
        CircularOrbit.fillHalfRow(1.0, deyV, jacobian[2], 3);
        double cle = u / this.a + this.ex - eSinE * beta * this.ey;
        double sle = v / this.a + this.ey + eSinE * beta * this.ex;
        double m1 = beta * eCosE;
        double m2 = 1.0 - m1 * eCosE;
        double m3 = u * this.ey - v * this.ex + eSinE * beta * (u * this.ex + v * this.ey);
        double m4 = -sle + cle * eSinE * beta;
        double m5 = cle + sle * eSinE * beta;
        CircularOrbit.fillHalfRow((2.0 * m3 / r + aOr * eSinE + m1 * eSinE * (1.0 + m1 - (1.0 + aOr) * m2) / epsilon) / r2, position, (m1 * m2 / epsilon - 1.0) * oOsqrtMuA, velocity, m4, dexP, m5, deyP, -sle / this.a, duP, cle / this.a, dvP, jacobian[5], 0);
        CircularOrbit.fillHalfRow((m1 * m2 / epsilon - 1.0) * oOsqrtMuA, position, (2.0 * m3 + eSinE * this.a + m1 * eSinE * r * (eCosE * beta * 2.0 - aOr * m2) / epsilon) / mu, velocity, m4, dexV, m5, deyV, -sle / this.a, duV, cle / this.a, dvV, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesian();
        double alphaE = this.getAlphaE();
        double cosAe = FastMath.cos((double)alphaE);
        double sinAe = FastMath.sin((double)alphaE);
        double aOr = 1.0 / (1.0 - this.ex * cosAe - this.ey * sinAe);
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = aOr * (rowL[j] + sinAe * rowEx[j] - cosAe * rowEy[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesian();
        double alphaE = this.getAlphaE();
        double cosAe = FastMath.cos((double)alphaE);
        double sinAe = FastMath.sin((double)alphaE);
        double eSinE = this.ex * sinAe - this.ey * cosAe;
        double ecosE = this.ex * cosAe + this.ey * sinAe;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double epsilon = FastMath.sqrt((double)(1.0 - e2));
        double onePeps = 1.0 + epsilon;
        double d = onePeps - ecosE;
        double cT = (d * d + eSinE * eSinE) / 2.0;
        double cE = ecosE * onePeps - e2;
        double cX = this.ex * eSinE / epsilon - this.ey + sinAe * onePeps;
        double cY = this.ey * eSinE / epsilon + this.ex - cosAe * onePeps;
        double factorLe = (cT + cE) / cT;
        double factorEx = cX / cT;
        double factorEy = cY / cT;
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowA = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowA[j] = factorLe * rowA[j] + factorEx * rowEx[j] + factorEy * rowEy[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngle type, double gm, double[] pDot) {
        double n = FastMath.sqrt((double)(gm / this.a)) / this.a;
        switch (type) {
            case MEAN: {
                pDot[5] = pDot[5] + n;
                break;
            }
            case ECCENTRIC: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos((double)this.alphaV) + this.ey * FastMath.sin((double)this.alphaV);
                pDot[5] = pDot[5] + n * ksi / oMe2;
                break;
            }
            case TRUE: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos((double)this.alphaV) + this.ey * FastMath.sin((double)this.alphaV);
                pDot[5] = pDot[5] + n * ksi * ksi / (oMe2 * FastMath.sqrt((double)oMe2));
                break;
            }
            default: {
                throw OrekitException.createInternalError(null);
            }
        }
    }

    public String toString() {
        return new StringBuffer().append("circular parameters: ").append('{').append("a: ").append(this.a).append(", ex: ").append(this.ex).append(", ey: ").append(this.ey).append(", i: ").append(FastMath.toDegrees((double)this.i)).append(", raan: ").append(FastMath.toDegrees((double)this.raan)).append(", alphaV: ").append(FastMath.toDegrees((double)this.alphaV)).append(";}").toString();
    }

    private Object writeReplace() {
        return new DTO(this);
    }

    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20140617L;
        private double[] d;
        private final Frame frame;

        private DTO(CircularOrbit orbit) {
            AbsoluteDate date = orbit.getDate();
            double epoch = FastMath.floor((double)date.durationFrom(AbsoluteDate.J2000_EPOCH));
            double offset = date.durationFrom(AbsoluteDate.J2000_EPOCH.shiftedBy(epoch));
            if (orbit.serializePV) {
                TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
                this.d = new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.i, orbit.raan, orbit.alphaV, pv.getPosition().getX(), pv.getPosition().getY(), pv.getPosition().getZ(), pv.getVelocity().getX(), pv.getVelocity().getY(), pv.getVelocity().getZ(), pv.getAcceleration().getX(), pv.getAcceleration().getY(), pv.getAcceleration().getZ()};
            } else {
                this.d = new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.i, orbit.raan, orbit.alphaV};
            }
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            if (this.d.length > 10) {
                return new CircularOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, new TimeStampedPVCoordinates(AbsoluteDate.J2000_EPOCH.shiftedBy(this.d[0]).shiftedBy(this.d[1]), new Vector3D(this.d[9], this.d[10], this.d[11]), new Vector3D(this.d[12], this.d[13], this.d[14]), new Vector3D(this.d[15], this.d[16], this.d[17])), this.frame, this.d[2]);
            }
            return new CircularOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, this.frame, AbsoluteDate.J2000_EPOCH.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

